/*----------------------------------------------------------------------
 *
 * FILE:  IMGHDR.H
 *
 * DESCRIPTION:
 *
 *  Here we define the Header Structure images in poppadom project
 *  History
 *
 *   Jason Hsue 11/21/2003      3rd Edition
 *      Add bit field structure members FamilySerialId and isUniversal.
 *   Jason Hsue 12/09/2003      4th Edition
 *      Add Multi image header and define module run time image and
 *      diag image type to support multi-image download.
 *----------------------------------------------------------------------*/

#ifndef _IMGHDR_H
#define _IMGHDR_H

/* INCLUDE FILE DECLARATIONS
 */

/* NAMING CONSTANT DECLARATIONS
 */
#define IMGHDR_IMAGE_LOADERTYPE                 0  /* not used in actual file */
#define IMGHDR_IMAGE_RUNTIMETYPE                1
#define IMGHDR_IMAGE_DIAGNOSTICTYPE             2
#define IMGHDR_IMAGE_MODULE_RUNTIMETYPE         3
#define IMGHDR_IMAGE_MODULE_DIAGNOSTICTYPE      4
#define IMGHDR_IMAGE_CPLDTYPE                   5

#define IMGHDR_MAGIC_PATTERN        0x1A2B3C4D
#define MULTI_IMGHDR_MAGIC_PATTERN  0x9A8B7C6D

#define	LDR_INFO_BLOCK_ID   "LDR0"

/* MACRO FUNCTION DECLARATIONS
 */
/* macro used to generate the "numOfBits" bits which are set as 1 from bit 0
 * the maximum number of bits to generate by this macro is 32 bits.
 */
#define SYS_IMGHDR_U32_BIT_MASK(numOfBits) (~(0xFFFFFFFFUL << (numOfBits)))

/* macro used to generate a not-bit-mask for the specfied field. The bits in
 * the specified "not-bit-mask" are set as 0, other bits are set as 1.
 *
 * field_offset - The leaset significant bit position of the field.
 * field_len    - The length of the field in bit
 */
#define SYS_IMGHDR_U32_BIT_FIELD_NOT_MASK(field_offset,field_len) \
    (~(SYS_IMGHDR_U32_BIT_MASK((field_len)) << (field_offset)))

/* macro used to generate a bit-mask for the specfied field. The bits in
 * the specified "bit-mask" are set as 1, other bits are set as 0.
 *
 * field_offset - The leaset significant bit position of the field.
 * field_len    - The length of the field in bit
 */
#define SYS_IMGHDR_U32_BIT_FIELD_MASK(field_offset,field_len) \
    ((SYS_IMGHDR_U32_BIT_MASK((field_len)) << (field_offset)))

/* macro used to set bit field in type UI32_T
 *
 * data   - The data in primitive data type which contains the bit field.
 * offset - The leaset significant bit position of the field.
 * length - The length of the field in bit
 * val    - The value to be set tho the specified bit field.
 */
#define SYS_IMGHDR_U32_SET_FIELD(data,offset,length,val) \
    (data) = (((data) & SYS_IMGHDR_U32_BIT_FIELD_NOT_MASK((offset),(length))) | ((val) <<(offset)))

/* macro used to get bit field value in type UI32_T
 *
 * data   - The data in primitive data type which contains the bit field.
 * offset - The leaset significant bit position of the field.
 * length - The length of the field in bit
 *
 * return value - the value in the specified bit field
 */
#define SYS_IMGHDR_U32_GET_FIELD(data,offset,length) \
        (((data) >> (offset)) & ((1UL << (length)) - 1UL))

/* Macros to set fields in IH_T
 */
#define SYS_IMGHDR_U32_SET_FIELD_IH_FAMILY_SERIAL_ID(ih, family_serial_id) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField1, 27, 5, (family_serial_id))
#define SYS_IMGHDR_U32_SET_FIELD_IH_IMAGE_TYPE(ih, image_type) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField1, 16, 11, (image_type))
#define SYS_IMGHDR_U32_SET_FIELD_IH_ZIP_TYPE(ih, zip_type) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField1, 0, 16, (zip_type))

#define SYS_IMGHDR_U32_SET_FIELD_IH_FILE_TYPE(ih, file_type) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField2, 24, 8, (file_type))
#define SYS_IMGHDR_U32_SET_FIELD_IH_IS_UNIVERSAL(ih, is_universal) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField2, 23, 1, (is_universal))
#define SYS_IMGHDR_U32_SET_FIELD_IH_YEAR(ih, year) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField2, 16, 7, (year))
#define SYS_IMGHDR_U32_SET_FIELD_IH_MONTH(ih, month) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField2, 8, 8, (month))
#define SYS_IMGHDR_U32_SET_FIELD_IH_DAY(ih, day) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField2, 0, 8, (day))

#define SYS_IMGHDR_U32_SET_FIELD_IH_TIME_HOUR(ih, time_hour) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField3, 24, 8, (time_hour))
#define SYS_IMGHDR_U32_SET_FIELD_IH_TIME_MINUTE(ih, time_minute) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField3, 16, 8, (time_minute))
#define SYS_IMGHDR_U32_SET_FIELD_IH_TIME_SECOND(ih, time_second) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField3, 8, 8, (time_second))
#define SYS_IMGHDR_U32_SET_FIELD_IH_PADDING(ih, padding) SYS_IMGHDR_U32_SET_FIELD((ih).MixedBitField3, 0, 8, (padding))

/* Macros to get fields in IH_T
 */
#define SYS_IMGHDR_U32_GET_FIELD_IH_FAMILY_SERIAL_ID(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField1, 27, 5)
#define SYS_IMGHDR_U32_GET_FIELD_IH_IMAGE_TYPE(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField1, 16, 11)
#define SYS_IMGHDR_U32_GET_FIELD_IH_ZIP_TYPE(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField1, 0, 16)

#define SYS_IMGHDR_U32_GET_FIELD_IH_FILE_TYPE(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField2, 24, 8)
#define SYS_IMGHDR_U32_GET_FIELD_IH_IS_UNIVERSAL(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField2, 23, 1)
#define SYS_IMGHDR_U32_GET_FIELD_IH_YEAR(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField2, 16, 7)
#define SYS_IMGHDR_U32_GET_FIELD_IH_MONTH(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField2, 8, 8)
#define SYS_IMGHDR_U32_GET_FIELD_IH_DAY(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField2, 0, 8)

#define SYS_IMGHDR_U32_GET_FIELD_IH_TIME_HOUR(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField3, 24, 8)
#define SYS_IMGHDR_U32_GET_FIELD_IH_TIME_MINUTE(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField3, 16, 8)
#define SYS_IMGHDR_U32_GET_FIELD_IH_TIME_SECOND(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField3, 8, 8)
#define SYS_IMGHDR_U32_GET_FIELD_IH_PADDING(ih) SYS_IMGHDR_U32_GET_FIELD((ih)->MixedBitField3, 0, 8)


/* DATA TYPE DECLARATIONS
 */
/* Each image instance contains this image header */
/* Assume byte postion of a long word is like B1B2B3B4, with B1 the Most-Significant-Byte */
typedef struct IH_S
{
    unsigned long  MagicWord;        /* IMGHDR_MAGIC_PATTERN */
    unsigned long  ImageChecksum;
    unsigned long  SWVersion;        /* B1.B2.B3:formal release number,B4:RD build nmuber */
    unsigned long  ImageLength;      /* the length does not include that of image header */
    /* The bit-field expression is difficult to write C code on machines that
     * use different endian. For ease of setting/getting these fields using C
     * code, these bit fields will be combined into a primitive data type field
     * and use macro to set and get these bit field. The original bit-field
     * definition is kept for ease of understanding bit field layout.
     *
     * Note that the value of primitive data type field alwas wriiten to the file
     * in big endian.
     * The C-style bit field declaration, which will be used as comment here,
     * always allocate the bits from the most siginficant bit to the leaset
     * significant bit. For example, the three bit field "FamilySerialId:5",
     * "ImageType:11" and "ZipType:16", which will be combined as a 32-bit
     * unsigned integer. The occupied bits for these three fields are shown
     * below:
     *     ZipType        -  Bit  0 - Bit 15
     *     ImageType      -  Bit 16 - Bit 26
     *     FamilySerialId -  Bit 27 - Bit 31
     *
     */
#if 0
    unsigned int   FamilySerialId  :5; /* A serial number for each family project          */
    unsigned int   ImageType       :11;/* Family project ID                                */
    unsigned int   ZipType         :16;/* one of ZIP_RAW, ZIP_GZIP, ZIP_PKZIP           */
#else
    unsigned int   MixedBitField1;
#endif

#if 0
    unsigned int   FileType        :8; /* file type of runtime/diagnostic                  */
    unsigned int   isUniversal     :1; /* The universal bit to identify universal image    */   
    unsigned int   Year            :7; /* year base on 2000 year                           */
    unsigned int   Month           :8; /* month                                            */
    unsigned int   Day             :8; /* day                                              */
#else
    unsigned int   MixedBitField2;
#endif

#if 0
    unsigned long  Time_Hour       :8; /* Image Release Time B1:hour(0..23),B2:minute(0..59),B3:second(0..59) */
    unsigned long  Time_Minute     :8;
    unsigned long  Time_Second     :8;
    unsigned long  Padding         :8;
#else
    unsigned int   MixedBitField3;
#endif

    unsigned int   SupportBidBitmap;/* bitmap for board ids that can apply this file */
    unsigned int   reserved[3];
    unsigned long  HeaderChecksum;
} IH_T;


/* This structure is a base of the binding image file header.
 * It's extendable but needs to be 16 bytes alignment.
 * For instance, if 2 images combine together, the header length is 48 bytes.
 * If 3 image combine together, it still is 48 bytes length.
 * But if 4 image combine together, it will be 64 bytes length.
 */
typedef struct MULTI_IH_S
{
    unsigned long   MagicWord;      /* A magic word to represent multi image header         */
    unsigned short  ImageNumber;    /* How many images combined in this file                */
    unsigned short  HeaderLength;   /* 48 basically.  It would be 16 bytes alignment        */
    unsigned long   Date;           /* File Release Date e.g. "20031205"                    */
    unsigned long   Time;           /* File Release Time B1:hour(0..23),B2:minute(0..59),B3:second(0..59) */

    unsigned long   HeaderChecksum; /* Header checksum                                      */
    unsigned long   FileVersion;    /* Combined file version will be the same as Image 1 version  */
    struct {
        unsigned long   Version;    /* The first image version, usually this is mainboard version */
        unsigned long   Length;     /* first image length, it should be 4 bytes alignment         */
    } Image[1];
} MULTI_IH_T;


/* Image Type constants defined here */
enum IMG_TYPE_E
{
   IMG_801 = 0,   /* 801 runtime*/
   IMG_302,       /* 302 prom code + runtime*/
   IMG_801_LDR,   /* Boot Loader*/
   IMG_801_PROM,  /* 801 prom code*/
   IMG_ALL        /*  All Banks of flash!*/
};

enum ZIP_TYPE_E
{
   ZIP_RAW = 0,   /* uncompressed*/
   ZIP_GZIP,      /* compressed with gzip*/
   ZIP_PKZIP,      /* compressed with pkzip*/
   ZIP_BZIP2
};

typedef struct LDR_INFO_BLOCK_S
{
    unsigned char   identifier[4];  /* should be assigned LDR_INFO_BLOCK_ID */
    unsigned char   version[32];     /* image version major.minor.fix.build */
    unsigned long   start_addr;     /* image start address in target */
    unsigned long   length;         /* image length */
    unsigned long   checksum;       /* image checksum */
    unsigned short  date_year;
    unsigned char   date_month;
    unsigned char   date_day;
    unsigned char   reserved[4];    /* reserved */
    unsigned long   block_checksum; /* checksum for the block excludes this field */
} LDR_INFO_BLOCK_T;

#endif /* end _IMGHDR_H */
